<?php
/**
 * Mobile Messages/Chat Interface
 * OBGYN Research & Collaboration Platform
 */

require_once '../config/config.php';

requireLogin();
if (!isResearcher()) {
    redirect(SITE_URL . '/index.php');
}

$pageTitle = 'Messages';
$userId = getCurrentUserId();

// Get user profile photo
$userPhoto = '/assets/images/default-avatar.png';
if ($userId) {
    $stmt = $db->prepare("SELECT photo FROM researcher_profiles WHERE user_id = ?");
    $stmt->execute([$userId]);
    $profileData = $stmt->fetch();
    $userPhoto = $profileData['photo'] ?? null;
    if ($userPhoto && !empty($userPhoto)) {
        $userPhoto = UPLOADS_URL . '/profile_pictures/' . (strpos($userPhoto, 'assets/uploads/profile_pictures/') !== false ? str_replace('assets/uploads/profile_pictures/', '', $userPhoto) : $userPhoto);
    } else {
        $userPhoto = '/assets/images/default-avatar.png';
    }
}
?>

<style>
/* Reset any conflicting styles first */
.mobile-messenger-container * {
    box-sizing: border-box;
}

/* Fix for header and footer overlay issues */
.mobile-messenger-container {
    position: fixed;
    top: 60px; /* Match your header height */
    left: 0;
    right: 0;
    bottom: 80px; /* Match your footer height */
    background: #fff;
    display: flex;
    flex-direction: column;
    overflow: hidden;
    z-index: 1000; /* Lower than header/footer */
    height: calc(100vh - 140px); /* 60px header + 80px footer */
}

/* Ensure header stays above */
.researcher-mobile-header {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    height: 60px;
    background: #fff;
    z-index: 1100 !important; /* Higher than messenger */
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

/* Ensure footer stays above */
.researcher-mobile-footer {
    position: fixed;
    bottom: 0;
    left: 0;
    right: 0;
    height: 80px;
    background: #fff;
    z-index: 1100 !important; /* Higher than messenger */
    box-shadow: 0 -2px 4px rgba(0,0,0,0.1);
}

/* Add padding to body to account for fixed header/footer */
body {
    padding-top: 60px !important;
    padding-bottom: 80px !important;
    overflow: hidden;
}

/* Make sure header/footer content is clickable */
.researcher-mobile-header *,
.researcher-mobile-footer * {
    pointer-events: auto !important;
    z-index: 1101 !important;
    position: relative;
}
/* Ensure all containers have proper stacking */
.conversation-list-view,
.chat-view {
    position: relative;
    width: 100%;
    height: 100%;
    background: white;
}

/* Conversation List View */
.conversation-list-view {
    display: flex;
    flex-direction: column;
    height: 100%;
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

.conversation-list-view.hidden {
    transform: translateX(-100%);
    position: absolute;
    width: 100%;
    pointer-events: none;
}

/* Conversation List */
.mobile-conversation-list {
    flex: 1;
    overflow-y: auto;
    -webkit-overflow-scrolling: touch;
    height: 100%;
}

.conversation-item {
    display: flex;
    align-items: center;
    padding: 12px 16px;
    border-bottom: 1px solid #f0f2f5;
    cursor: pointer;
    transition: background-color 0.2s;
    position: relative;
    background: white;
    user-select: none;
    -webkit-user-select: none;
    -webkit-tap-highlight-color: transparent;
}

.conversation-item:active {
    background-color: #f0f2f5 !important;
}

.conversation-item.unread {
    background-color: #f8f9fa;
}

.conversation-avatar {
    width: 56px;
    height: 56px;
    border-radius: 50%;
    margin-right: 12px;
    flex-shrink: 0;
    object-fit: cover;
}

.conversation-avatar-placeholder {
    width: 56px;
    height: 56px;
    border-radius: 50%;
    margin-right: 12px;
    flex-shrink: 0;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 1.2rem;
}

/* Group Avatar Styles for Mobile */
.group-avatar-container {
    width: 56px;
    height: 56px;
    margin-right: 12px;
    flex-shrink: 0;
    position: relative;
    display: flex;
    align-items: center;
    justify-content: center;
}

.group-avatar-item {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    border: 2px solid white;
    object-fit: cover;
    position: absolute;
}

.group-avatar-item:nth-child(1) {
    top: 0;
    left: 0;
    z-index: 3;
}

.group-avatar-item:nth-child(2) {
    top: 0;
    right: 0;
    z-index: 2;
}

.group-avatar-item:nth-child(3) {
    bottom: 0;
    left: 50%;
    transform: translateX(-50%);
    z-index: 1;
}

.group-avatar-extra {
    position: absolute;
    bottom: 0;
    right: 0;
    background: #6c757d;
    color: white;
    border-radius: 50%;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 0.7rem;
    font-weight: 600;
    border: 2px solid white;
    z-index: 4;
}

/* Search Styles */
.conversation-search-container {
    padding: 12px 16px;
    border-bottom: 1px solid #f0f2f5;
    background: #fff;
    position: relative;
    z-index: 1;
}

.conversation-search-input {
    width: 100%;
    padding: 10px 16px;
    border: 1px solid #e9ecef;
    border-radius: 20px;
    background: #f0f2f5;
    font-size: 0.95rem;
}

.conversation-search-input:focus {
    outline: none;
    background: #fff;
    border-color: #0d6efd;
}

.conversation-content {
    flex: 1;
    min-width: 0;
}

.conversation-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 4px;
}

.conversation-name {
    font-weight: 600;
    font-size: 0.95rem;
    color: #050505;
    margin: 0;
}

.conversation-time {
    font-size: 0.8rem;
    color: #65676b;
}

.conversation-preview {
    font-size: 0.9rem;
    color: #65676b;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
    margin: 0;
}

.conversation-item.unread .conversation-preview {
    font-weight: 600;
    color: #050505;
}

.unread-badge {
    position: absolute;
    right: 16px;
    bottom: 16px;
    background: #0d6efd;
    color: white;
    border-radius: 12px;
    padding: 2px 8px;
    font-size: 0.75rem;
    font-weight: 600;
}

/* Chat View */
.chat-view {
    display: flex;
    flex-direction: column;
    height: 100%;
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: #fff;
    transform: translateX(100%);
    transition: transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    z-index: 2;
}

.chat-view.active {
    transform: translateX(0);
    z-index: 3;
}

/* Chat Header - CRITICAL FIXES */
.chat-header {
    display: flex;
    align-items: center;
    padding: 12px 16px;
    border-bottom: 1px solid #e9ecef;
    background: #fff;
    position: relative;
    z-index: 100;
    min-height: 60px;
}

.chat-back-btn {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    background: transparent;
    border: none;
    color: #050505;
    margin-right: 8px;
    cursor: pointer;
    flex-shrink: 0;
    -webkit-tap-highlight-color: transparent;
    touch-action: manipulation;
    position: relative;
    z-index: 101;
}

.chat-back-btn:active {
    background: #f0f2f5;
}

.chat-user-info {
    flex: 1;
    display: flex;
    align-items: center;
    cursor: pointer;
    padding: 8px 12px;
    border-radius: 8px;
    transition: background-color 0.2s;
    -webkit-tap-highlight-color: transparent;
    min-height: 44px; /* Minimum touch target size */
    position: relative;
    z-index: 1;
}

.chat-user-info:active {
    background-color: #f0f2f5;
}

.chat-actions {
    display: flex;
    gap: 8px;
    margin-left: 8px;
    position: relative;
    z-index: 102;
}

.chat-action-btn {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    background: transparent;
    border: none;
    color: #6c757d;
    cursor: pointer;
    transition: all 0.2s;
    flex-shrink: 0;
    -webkit-tap-highlight-color: transparent;
    touch-action: manipulation;
    position: relative;
    z-index: 103;
}

.chat-action-btn:active {
    background: #f0f2f5;
    transform: scale(0.95);
}

.chat-user-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 10px;
    object-fit: cover;
    flex-shrink: 0;
}

.chat-user-avatar-placeholder {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 10px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    flex-shrink: 0;
}

.chat-user-name {
    font-weight: 600;
    font-size: 1rem;
    color: #050505;
    margin: 0;
}

/* Messages Area */
.messages-container {
    flex: 1;
    overflow-y: auto;
    -webkit-overflow-scrolling: touch;
    padding: 16px;
    background: #fff;
    position: relative;
    z-index: 1;
}

.message-input-container {
    padding: 12px 16px;
    background: #fff;
    border-top: 1px solid #e9ecef;
    display: flex;
    align-items: flex-end;
    gap: 8px;
    position: relative;
    z-index: 10;
}

.message-input-wrapper {
    flex: 1;
    position: relative;
}

.message-input {
    width: 100%;
    min-height: 40px;
    max-height: 120px;
    padding: 10px 16px;
    border: 1px solid #e9ecef;
    border-radius: 20px;
    background: #f0f2f5;
    resize: none;
    font-size: 0.95rem;
    line-height: 1.4;
    overflow-y: auto;
}

.message-input:focus {
    outline: none;
    background: #fff;
    border-color: #0d6efd;
}

.send-button {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #0d6efd;
    border: none;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    cursor: pointer;
    flex-shrink: 0;
    transition: transform 0.2s;
    -webkit-tap-highlight-color: transparent;
    touch-action: manipulation;
}

.send-button:active {
    transform: scale(0.95);
}

/* Empty States */
.empty-state {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    height: 100%;
    padding: 32px;
    text-align: center;
}

/* Mobile Modal Styles - CRITICAL FIXES */
.mobile-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255, 255, 255, 0.7);
    backdrop-filter: blur(10px);
    display: flex;
    align-items: flex-end;
    z-index: 9999;
    opacity: 0;
    pointer-events: none;
    transition: opacity 0.3s ease;
}

.mobile-modal-overlay.show {
    opacity: 1;
    pointer-events: auto;
}

.mobile-modal-content {
    background: #fff;
    border-radius: 16px 16px 0 0;
    width: 100%;
    max-height: 80vh;
    overflow-y: auto;
    -webkit-overflow-scrolling: touch;
    transform: translateY(100%);
    transition: transform 0.3s ease;
    pointer-events: auto;
}

.mobile-modal-overlay.show .mobile-modal-content {
    transform: translateY(0);
}

.mobile-modal-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 16px 20px;
    border-bottom: 1px solid #e9ecef;
    position: relative;
    z-index: 1;
}

.mobile-modal-header .btn-close {
    background: none;
    border: none;
    font-size: 1.2rem;
    color: #6c757d;
    cursor: pointer;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    -webkit-tap-highlight-color: transparent;
}

.mobile-modal-body {
    padding: 20px;
    position: relative;
    z-index: 1;
}

/* Participants Styles */
.participant-item {
    display: flex;
    align-items: center;
    padding: 12px 0;
    border-bottom: 1px solid #f8f9fa;
}

.participant-item:last-child {
    border-bottom: none;
}

.participant-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    margin-right: 12px;
    flex-shrink: 0;
}

.participant-avatar img {
    width: 100%;
    height: 100%;
    border-radius: 50%;
    object-fit: cover;
}

.avatar-placeholder {
    width: 100%;
    height: 100%;
    border-radius: 50%;
    background: #e9ecef;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
    font-weight: 600;
    font-size: 14px;
}

.participant-info {
    flex: 1;
    min-width: 0;
}

.participant-name {
    font-weight: 500;
    margin-bottom: 2px;
}

.participant-role {
    font-size: 12px;
    color: #6c757d;
}

.participant-actions {
    display: flex;
    gap: 8px;
    margin-left: 12px;
}

.participant-actions button {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    border: none;
    cursor: pointer;
    -webkit-tap-highlight-color: transparent;
}

/* Search Results for Mobile */
.search-results {
    max-height: 200px;
    overflow-y: auto;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
    background: white;
    margin-top: 8px;
}

.search-result-item {
    display: flex;
    align-items: center;
    padding: 12px;
    cursor: pointer;
    border-bottom: 1px solid #f8f9fa;
    transition: background-color 0.2s;
    -webkit-tap-highlight-color: transparent;
    min-height: 44px;
}

.search-result-item:active {
    background-color: #f8f9fa;
}

.search-result-item:last-child {
    border-bottom: none;
}

.search-result-avatar {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    margin-right: 12px;
    object-fit: cover;
    flex-shrink: 0;
}

.search-result-avatar-placeholder {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    margin-right: 12px;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 12px;
    flex-shrink: 0;
}

.search-result-info h6 {
    margin: 0;
    font-size: 14px;
    font-weight: 600;
}

.search-result-info small {
    color: #6c757d;
    font-size: 12px;
}

.search-result-item.selected {
    background-color: #e3f2fd;
    border-color: #2196f3;
}

/* Alert Styles */
.mobile-alert-container {
    position: fixed;
    top: 70px;
    left: 16px;
    right: 16px;
    z-index: 10000;
    pointer-events: none;
}

.mobile-alert {
    padding: 12px 16px;
    border-radius: 8px;
    margin-bottom: 8px;
    font-size: 14px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    animation: slideDown 0.3s ease;
    pointer-events: auto;
}

@keyframes slideDown {
    from {
        transform: translateY(-100%);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

/* Skeleton Loading */
.skeleton-conversation {
    display: flex;
    align-items: center;
    padding: 12px 16px;
    border-bottom: 1px solid #f0f2f5;
}

/* Message bubbles and other existing styles remain the same */
.message-group {
    margin-bottom: 16px;
}

.message-item {
    display: flex;
    margin-bottom: 8px;
    animation: messageSlideIn 0.3s ease-out;
}

@keyframes messageSlideIn {
    from {
        opacity: 0;
        transform: translateY(10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.message-item.sent {
    justify-content: flex-end;
}

.message-bubble {
    max-width: 75%;
    padding: 10px 14px;
    border-radius: 18px;
    word-wrap: break-word;
}

.message-item.received .message-bubble {
    background: #f0f2f5;
    color: #050505;
    border-bottom-left-radius: 4px;
}

.message-item.sent .message-bubble {
    background: #0d6efd;
    color: white;
    border-bottom-right-radius: 4px;
}

.message-time {
    font-size: 0.7rem;
    color: #65676b;
    margin-top: 4px;
    text-align: right;
}

.message-item.sent .message-time {
    color: rgba(255, 255, 255, 0.8);
}

.system-message {
    display: flex;
    justify-content: center;
    margin: 8px 0;
}

.system-message-text {
    background: #f0f2f5;
    color: #65676b;
    padding: 6px 12px;
    border-radius: 12px;
    font-size: 0.8rem;
    text-align: center;
    max-width: 80%;
}

.message-sender {
    font-size: 0.8rem;
    font-weight: 600;
    color: #050505;
    margin-bottom: 4px;
}

.message-text {
    margin-bottom: 4px;
}
</style>

<div class="mobile-messenger-container">
    <!-- Conversation List View -->
    <div class="conversation-list-view" id="conversationListView">
        <!-- Search Bar -->
        <div class="conversation-search-container">
            <input type="text" id="mobile-chat-search" class="conversation-search-input" placeholder="Search conversations...">
        </div>
        
        <div class="mobile-conversation-list" id="mobileConversationList">
            <!-- Loading skeleton -->
            <div id="conversationListSkeleton">
                <?php for ($i = 0; $i < 5; $i++): ?>
                <div class="skeleton-conversation">
                    <div class="skeleton-avatar"></div>
                    <div class="skeleton-content">
                        <div class="skeleton-line"></div>
                        <div class="skeleton-line short"></div>
                    </div>
                </div>
                <?php endfor; ?>
            </div>
        </div>
    </div>

    <!-- Chat View -->
    <div class="chat-view" id="chatView">
        <!-- Chat Header -->
        <div class="chat-header" id="chatHeader">
            <button class="chat-back-btn" onclick="MobileChatManager.closeChat()">
                <i class="fas fa-arrow-left"></i>
            </button>
            <div class="chat-user-info" id="chatUserInfo" onclick="MobileChatManager.showParticipantsModal()">
                <!-- Will be populated dynamically -->
            </div>
            <div class="chat-actions" id="chatActions">
                <button class="chat-action-btn" onclick="MobileChatManager.showAddParticipantsModal()" title="Add Participants">
                    <i class="fas fa-user-plus"></i>
                </button>
            </div>
        </div>

        <!-- Messages Container -->
        <div class="messages-container" id="messagesContainer">
            <div class="empty-state">
                <i class="fas fa-comments"></i>
                <h3>No messages yet</h3>
                <p>Start the conversation!</p>
            </div>
        </div>

        <!-- Message Input -->
        <div class="message-input-container" id="messageInputContainer">
            <div class="message-input-wrapper">
                <textarea 
                    id="messageInput" 
                    class="message-input" 
                    placeholder="Type a message..."
                    rows="1"
                ></textarea>
            </div>
            <button class="send-button" id="sendButton" onclick="MobileChatManager.sendMessage()">
                <i class="fas fa-paper-plane"></i>
            </button>
        </div>
    </div>
</div>

<!-- Alert Container for Mobile -->
<div class="mobile-alert-container" id="mobileAlertContainer"></div>

<!-- Mobile Modals -->
<!-- Participants Modal -->
<div class="mobile-modal-overlay" id="participantsModalOverlay">
    <div class="mobile-modal-content">
        <div class="mobile-modal-header">
            <h5 class="mb-0">Chat Participants</h5>
            <button type="button" class="btn-close" onclick="MobileChatManager.hideParticipantsModal()">&times;</button>
        </div>
        <div class="mobile-modal-body">
            <div id="mobile-participants-list">
                <!-- Participants will be populated here -->
            </div>
        </div>
    </div>
</div>

<!-- Add Participants Modal -->
<div class="mobile-modal-overlay" id="addParticipantsModalOverlay">
    <div class="mobile-modal-content">
        <div class="mobile-modal-header">
            <h5 class="mb-0">Add Participants</h5>
            <button type="button" class="btn-close" onclick="MobileChatManager.hideAddParticipantsModal()">&times;</button>
        </div>
        <div class="mobile-modal-body">
            <div class="mb-3">
                <input type="text" id="mobile-add-participant-search" class="form-control" placeholder="Search researchers...">
            </div>
            <div id="mobile-add-participant-results" class="search-results">
                <div class="text-center text-muted py-3">Loading researchers...</div>
            </div>
            <div class="alert alert-info mt-3">
                <i class="fas fa-info-circle"></i> Adding a participant will convert this conversation to a group chat.
            </div>
        </div>
    </div>
</div>

<!-- New Message Modal -->
<div class="mobile-modal-overlay" id="newMessageModalOverlay">
    <div class="mobile-modal-content">
        <div class="mobile-modal-header">
            <h5 class="mb-0">New Message</h5>
            <button type="button" class="btn-close" onclick="MobileChatManager.hideNewMessageModal()">&times;</button>
        </div>
        <div class="mobile-modal-body">
            <div class="mb-3">
                <input type="text" id="mobile-new-message-search" class="form-control" placeholder="Search researchers...">
            </div>
            <div id="mobile-new-message-results" class="search-results">
                <div class="text-center text-muted py-3">Loading researchers...</div>
            </div>
            <div class="mb-3">
                <textarea id="mobile-initial-message" class="form-control" rows="3" placeholder="Type your first message..."></textarea>
            </div>
            <div class="d-flex justify-content-end gap-2">
                <button type="button" class="btn btn-secondary" onclick="MobileChatManager.hideNewMessageModal()">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="MobileChatManager.sendNewMessage()">Send Message</button>
            </div>
        </div>
    </div>
</div>

<script>
// Mobile Chat Manager
const MobileChatManager = {
    chatRooms: [],
    currentRoomId: null,
    currentRoomName: '',
    currentRoomParticipants: [],
    messages: [],
    lastMessageId: null,
    
// In MobileChatManager init, add:
init() {
    console.log('MobileChatManager initializing...');
    
    // Ensure header and footer are above our container
    this.fixHeaderFooterZIndex();
    
    this.setupEventListeners();
    this.loadConversations();
    
    // Force remove any conflicting overlays
    this.cleanupOverlays();
},

fixHeaderFooterZIndex() {
    // Find and fix header
    const header = document.querySelector('.researcher-mobile-header, [class*="header"], [class*="navbar"]');
    if (header) {
        header.style.zIndex = '1100';
        header.style.position = 'fixed';
        header.style.top = '0';
        header.style.left = '0';
        header.style.right = '0';
        console.log('Header z-index fixed');
    }
    
    // Find and fix footer  
    const footer = document.querySelector('.researcher-mobile-footer, [class*="footer"], [class*="bottom-nav"]');
    if (footer) {
        footer.style.zIndex = '1100';
        footer.style.position = 'fixed';
        footer.style.bottom = '0';
        footer.style.left = '0';
        footer.style.right = '0';
        console.log('Footer z-index fixed');
    }
    
    // Add click event listeners to header/footer
    this.setupHeaderFooterEvents();
},

setupHeaderFooterEvents() {
    // Ensure header buttons are clickable
    const headerButtons = document.querySelectorAll('.researcher-mobile-header button, .researcher-mobile-header a');
    headerButtons.forEach(button => {
        button.style.zIndex = '1101';
        button.style.position = 'relative';
    });
    
    // Ensure footer buttons are clickable
    const footerButtons = document.querySelectorAll('.researcher-mobile-footer button, .researcher-mobile-footer a');
    footerButtons.forEach(button => {
        button.style.zIndex = '1101';
        button.style.position = 'relative';
    });
},
    
    cleanupOverlays() {
        // Remove any Bootstrap modal backdrops that might be blocking
        const backdrops = document.querySelectorAll('.modal-backdrop');
        backdrops.forEach(backdrop => backdrop.remove());
        
        // Hide any other modals that might be open
        const modals = document.querySelectorAll('.modal');
        modals.forEach(modal => {
            if (modal.style.display === 'block' || modal.classList.contains('show')) {
                modal.style.display = 'none';
                modal.classList.remove('show');
            }
        });
        
        // Ensure body has proper styling
        document.body.style.overflow = 'hidden';
        document.body.style.position = 'relative';
    },
    
    setupEventListeners() {
        console.log('Setting up event listeners...');
        
        // Message input handling
        const messageInput = document.getElementById('messageInput');
        if (messageInput) {
            messageInput.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.sendMessage();
                }
            });
        }
        
        // Send button
        const sendButton = document.getElementById('sendButton');
        if (sendButton) {
            sendButton.addEventListener('click', () => this.sendMessage());
        }
        
        // Chat search
        const chatSearch = document.getElementById('mobile-chat-search');
        if (chatSearch) {
            chatSearch.addEventListener('input', (e) => {
                this.searchChats(e.target.value);
            });
        }
        
        // Add participant search
        const addParticipantSearch = document.getElementById('mobile-add-participant-search');
        if (addParticipantSearch) {
            addParticipantSearch.addEventListener('input', (e) => {
                this.searchResearchers(e.target.value);
            });
        }
        
        // New message search
        const newMessageSearch = document.getElementById('mobile-new-message-search');
        if (newMessageSearch) {
            newMessageSearch.addEventListener('input', (e) => {
                this.searchResearchersForNewMessage(e.target.value);
            });
        }
        
        // Add click event for conversation items (delegated)
        const conversationList = document.getElementById('mobileConversationList');
        if (conversationList) {
            conversationList.addEventListener('click', (e) => {
                const conversationItem = e.target.closest('.conversation-item');
                if (conversationItem) {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('Conversation item clicked via delegation');
                    // Extract room info from data attributes or find another way
                    const roomId = conversationItem.getAttribute('data-room-id');
                    const roomName = conversationItem.getAttribute('data-room-name');
                    if (roomId && roomName) {
                        this.openChat(parseInt(roomId), roomName);
                    }
                }
            });
        }
        
        console.log('Event listeners setup complete');
    },
    
    async loadConversations() {
        try {
            const response = await fetch('../api/chat.php?action=get_rooms');
            const data = await response.json();
            
            if (data.success) {
                this.chatRooms = data.rooms || [];
                this.renderConversationList();
            } else {
                this.renderConversationList(); // Show empty state
            }
        } catch (error) {
            console.error('Error loading conversations:', error);
            this.renderConversationList(); // Show empty state
        }
    },
    
    renderConversationList() {
        const container = document.getElementById('mobileConversationList');
        if (!container) {
            console.error('Conversation list container not found');
            return;
        }
        
        // Hide skeleton
        const skeleton = document.getElementById('conversationListSkeleton');
        if (skeleton) {
            skeleton.style.display = 'none';
        }
        
        if (this.chatRooms.length === 0) {
            container.innerHTML = '<div class="empty-state"><h3>No conversations</h3><p>Start a new conversation to begin chatting</p><button class="btn btn-primary" onclick="MobileChatManager.showNewChatModal()">Start Chat</button></div>';
            return;
        }
        
        let html = '';
        for (let i = 0; i < this.chatRooms.length; i++) {
            const room = this.chatRooms[i];
            const isGroup = room.participants && room.participants.length > 1;
            const roomName = room.name || (isGroup ? 'Group Chat' : (room.participants && room.participants.length > 0 ? 
                room.participants[0].first_name + ' ' + room.participants[0].last_name : 'Unknown'));
            const lastMessage = room.last_message || 'No messages';
            const unreadCount = room.unread_count || 0;
            
            // Generate avatar HTML
            let avatarHtml = '';
            if (isGroup && room.participants.length > 1) {
                const displayParticipants = room.participants.slice(0, 3);
                avatarHtml = `<div class="group-avatar-container">`;
                displayParticipants.forEach((participant, index) => {
                    if (participant.photo) {
                        avatarHtml += `<img src="${participant.photo}" class="group-avatar-item" alt="${this.escapeHtml(participant.first_name)}">`;
                    } else {
                        const initials = this.getInitials(participant.first_name + ' ' + participant.last_name);
                        avatarHtml += `<div class="group-avatar-item conversation-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${initials}</div>`;
                    }
                });
                if (room.participants.length > 3) {
                    avatarHtml += `<div class="group-avatar-extra">+${room.participants.length - 3}</div>`;
                }
                avatarHtml += `</div>`;
            } else {
                const participant = room.participants && room.participants.length > 0 ? room.participants[0] : null;
                if (participant && participant.photo) {
                    avatarHtml = `<img src="${participant.photo}" class="conversation-avatar" alt="${this.escapeHtml(participant.first_name)}">`;
                } else {
                    const name = participant ? participant.first_name + ' ' + participant.last_name : 'Unknown';
                    const initials = this.getInitials(name);
                    avatarHtml = `<div class="conversation-avatar-placeholder" style="background: ${this.stringToColor(name)}">${initials}</div>`;
                }
            }
            
            html += `<div class="conversation-item" data-room-id="${room.id}" data-room-name="${this.escapeHtml(roomName)}">`;
            if (unreadCount > 0) {
                html += ' unread';
            }
            
            html += avatarHtml;
            
            html += `<div class="conversation-content">`;
            html += `<div class="conversation-header">`;
            html += `<h3 class="conversation-name">${roomName}</h3>`;
            html += `<span class="conversation-time">${room.last_message_time ? this.timeAgo(room.last_message_time) : ''}</span>`;
            html += `</div>`;
            html += `<p class="conversation-preview">${lastMessage}</p>`;
            if (isGroup) {
                html += `<small class="text-muted">${room.participants.length} participants</small>`;
            }
            html += `</div>`;
            
            if (unreadCount > 0) {
                html += `<span class="unread-badge">${unreadCount}</span>`;
            }
            
            html += `</div>`;
        }
        
        container.innerHTML = html;
        console.log('Conversation list rendered with', this.chatRooms.length, 'items');
    },
    
    searchChats(query) {
        const container = document.getElementById('mobileConversationList');
        const items = container.querySelectorAll('.conversation-item');
        
        if (!query.trim()) {
            items.forEach(item => item.style.display = 'flex');
            return;
        }
        
        const searchTerm = query.toLowerCase();
        items.forEach(item => {
            const text = item.textContent.toLowerCase();
            if (text.includes(searchTerm)) {
                item.style.display = 'flex';
            } else {
                item.style.display = 'none';
            }
        });
    },
    
    async openChat(roomId, roomName) {
        console.log('Opening chat:', roomId, roomName);
        this.currentRoomId = roomId;
        this.currentRoomName = roomName;
        
        // Switch to chat view
        const conversationView = document.getElementById('conversationListView');
        const chatView = document.getElementById('chatView');
        
        if (conversationView && chatView) {
            conversationView.classList.add('hidden');
            chatView.classList.add('active');
            console.log('Views switched successfully');
        } else {
            console.error('Could not find conversationListView or chatView elements');
        }
        
        // Load room details including participants
        await this.loadRoomDetails();
        
        // Update chat header
        this.updateChatHeader();
        
        // Load messages
        await this.loadMessages();
    },
    
    async loadRoomDetails() {
        if (!this.currentRoomId) return;
        
        try {
            const response = await fetch(`../api/chat.php?action=get_room_details&room_id=${this.currentRoomId}`);
            const data = await response.json();
            
            if (data.success) {
                this.currentRoomParticipants = data.participants || [];
            }
        } catch (error) {
            console.error('Error loading room details:', error);
        }
    },
    
    updateChatHeader() {
        const userInfo = document.getElementById('chatUserInfo');
        if (!userInfo) return;
        
        if (!this.currentRoomId) {
            userInfo.innerHTML = '<h6 class="mb-0">Select a conversation</h6>';
            return;
        }
        
        const isGroup = this.currentRoomParticipants.length > 1;
        
        if (isGroup) {
            // Group chat header with multiple avatars
            const displayParticipants = this.currentRoomParticipants.slice(0, 3);
            let avatarsHtml = '<div class="group-avatar-container" style="width: 40px; height: 40px;">';
            displayParticipants.forEach((participant, index) => {
                if (participant.photo) {
                    avatarsHtml += `<img src="${participant.photo}" class="group-avatar-item" style="width: 20px; height: 20px;" alt="${this.escapeHtml(participant.first_name)}">`;
                } else {
                    const initials = this.getInitials(participant.first_name + ' ' + participant.last_name);
                    avatarsHtml += `<div class="group-avatar-item conversation-avatar-placeholder" style="width: 20px; height: 20px; font-size: 10px; background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${initials}</div>`;
                }
            });
            if (this.currentRoomParticipants.length > 3) {
                avatarsHtml += `<div class="group-avatar-extra" style="width: 16px; height: 16px; font-size: 8px;">+${this.currentRoomParticipants.length - 3}</div>`;
            }
            avatarsHtml += '</div>';
            
            userInfo.innerHTML = `
                ${avatarsHtml}
                <div>
                    <h6 class="chat-user-name">${this.escapeHtml(this.currentRoomName)}</h6>
                    <small class="text-muted">${this.currentRoomParticipants.length} participants</small>
                </div>
            `;
        } else {
            // Direct message header
            const participant = this.currentRoomParticipants[0] || {};
            const avatarHtml = participant.photo ? 
                `<img src="${participant.photo}" class="chat-user-avatar" alt="${this.escapeHtml(participant.first_name)}">` :
                `<div class="chat-user-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
            
            userInfo.innerHTML = `
                ${avatarHtml}
                <h6 class="chat-user-name">${this.escapeHtml(this.currentRoomName)}</h6>
            `;
        }
    },
    
    showParticipantsModal() {
        console.log('Showing participants modal');
        if (!this.currentRoomId) return;
        
        const modal = document.getElementById('participantsModalOverlay');
        
        // Populate participants list
        const participantsList = document.getElementById('mobile-participants-list');
        participantsList.innerHTML = this.currentRoomParticipants.map(participant => {
            const isCurrentUser = participant.id == <?php echo $userId; ?>;
            const avatarHtml = participant.photo ? 
                `<img src="${participant.photo}" class="participant-avatar" alt="${this.escapeHtml(participant.first_name)}">` :
                `<div class="participant-avatar-placeholder" style="background: ${this.stringToColor(participant.first_name + ' ' + participant.last_name)}">${this.getInitials(participant.first_name + ' ' + participant.last_name)}</div>`;
            
            return `
                <div class="participant-item">
                    ${avatarHtml}
                    <div class="participant-info">
                        <div class="participant-name">${this.escapeHtml(participant.first_name + ' ' + participant.last_name)}</div>
                        <div class="participant-role">${participant.role} ${isCurrentUser ? '(You)' : ''}</div>
                    </div>
                    ${!isCurrentUser ? `
                    <div class="participant-actions">
                        ${participant.role !== 'admin' ? `
                        <button class="btn btn-sm btn-success" onclick="MobileChatManager.makeAdmin(${participant.id})" title="Make Admin">
                            <i class="fas fa-user-shield"></i>
                        </button>
                        ` : ''}
                        <button class="btn btn-sm btn-danger" onclick="MobileChatManager.removeParticipant(${participant.id}, '${this.escapeHtml(participant.first_name + ' ' + participant.last_name).replace(/'/g, "\\'")}')" title="Remove">
                            <i class="fas fa-user-minus"></i>
                        </button>
                    </div>
                    ` : ''}
                </div>
            `;
        }).join('');
        
        modal.classList.add('show');
    },
    
    hideParticipantsModal() {
        const modal = document.getElementById('participantsModalOverlay');
        modal.classList.remove('show');
    },
    
    showAddParticipantsModal() {
        console.log('Showing add participants modal');
        if (!this.currentRoomId) return;
        
        const modal = document.getElementById('addParticipantsModalOverlay');
        this.loadResearchersForAdding();
        modal.classList.add('show');
    },
    
    hideAddParticipantsModal() {
        const modal = document.getElementById('addParticipantsModalOverlay');
        modal.classList.remove('show');
    },
    
    showNewMessageModal() {
        const modal = document.getElementById('newMessageModalOverlay');
        this.loadResearchersForNewMessage();
        modal.classList.add('show');
    },
    
    hideNewMessageModal() {
        const modal = document.getElementById('newMessageModalOverlay');
        modal.classList.remove('show');
    },
    
    async loadResearchersForAdding() {
        try {
            const response = await fetch('../api/search.php?q=&type=researchers&limit=100');
            const data = await response.json();
            
            const container = document.getElementById('mobile-add-participant-results');
            if (data.success && data.results) {
                // Filter out users already in the current room
                const currentParticipants = this.chatRooms.find(room => room.id == this.currentRoomId)?.participants || [];
                const currentUserIds = currentParticipants.map(p => p.id);
                
                const availableUsers = data.results.filter(user => !currentUserIds.includes(user.id));
                
                if (availableUsers.length === 0) {
                    container.innerHTML = '<div class="text-center text-muted py-3">No researchers available to add</div>';
                } else {
                    container.innerHTML = availableUsers.map(user => {
                        const letterAvatarHtml = `<div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>`;
                        const avatarHtml = user.photo ? 
                            `<img src="${user.photo}" class="search-result-avatar" alt="${this.escapeHtml(user.first_name)}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}; display: none;">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>` :
                            letterAvatarHtml;
                        
                        return `
                            <div class="search-result-item" onclick="MobileChatManager.addParticipant(${user.id})">
                                ${avatarHtml}
                                <div class="search-result-info">
                                    <h6 class="mb-0">${this.escapeHtml(user.first_name + ' ' + user.last_name)}</h6>
                                    <small class="text-muted">${user.institution || ''}</small>
                                </div>
                            </div>
                        `;
                    }).join('');
                }
            } else {
                container.innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
            }
        } catch (error) {
            console.error('Error:', error);
            document.getElementById('mobile-add-participant-results').innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
        }
    },
    
    searchResearchers(query) {
        const container = document.getElementById('mobile-add-participant-results');
        const items = container.querySelectorAll('.search-result-item');
        
        if (!query.trim()) {
            // Show all when search is empty
            this.loadResearchersForAdding();
            return;
        }
        
        const searchTerm = query.toLowerCase();
        let hasVisibleItems = false;
        
        items.forEach(item => {
            const text = item.textContent.toLowerCase();
            if (text.includes(searchTerm)) {
                item.style.display = 'flex';
                hasVisibleItems = true;
            } else {
                item.style.display = 'none';
            }
        });
        
        if (!hasVisibleItems) {
            container.innerHTML = '<div class="text-center text-muted py-3">No researchers found</div>';
        }
    },
    
    async loadResearchersForNewMessage() {
        try {
            const response = await fetch('../api/search.php?q=&type=researchers&limit=100');
            const data = await response.json();
            
            const container = document.getElementById('mobile-new-message-results');
            if (data.success && data.results) {
                // Filter out current user
                const currentUserId = <?php echo $userId; ?>;
                const availableUsers = data.results.filter(user => user.id != currentUserId);
                
                if (availableUsers.length === 0) {
                    container.innerHTML = '<div class="text-center text-muted py-3">No researchers available</div>';
                } else {
                    container.innerHTML = availableUsers.map(user => {
                        const letterAvatarHtml = `<div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>`;
                        const avatarHtml = user.photo ? 
                            `<img src="${user.photo}" class="search-result-avatar" alt="${this.escapeHtml(user.first_name)}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"><div class="search-result-avatar-placeholder" style="background: ${this.stringToColor(user.first_name + ' ' + user.last_name)}; display: none;">${this.getInitials(user.first_name + ' ' + user.last_name)}</div>` :
                            letterAvatarHtml;
                        
                        return `
                            <div class="search-result-item" onclick="MobileChatManager.selectResearcherForNewMessage(${user.id}, '${this.escapeHtml(user.first_name + ' ' + user.last_name).replace(/'/g, "\\'")}')">
                                ${avatarHtml}
                                <div class="search-result-info">
                                    <h6 class="mb-0">${this.escapeHtml(user.first_name + ' ' + user.last_name)}</h6>
                                    <small class="text-muted">${user.institution || ''}</small>
                                </div>
                            </div>
                        `;
                    }).join('');
                }
            } else {
                container.innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
            }
        } catch (error) {
            console.error('Error:', error);
            document.getElementById('mobile-new-message-results').innerHTML = '<div class="text-center text-muted py-3">Error loading researchers</div>';
        }
    },
    
    selectResearcherForNewMessage(userId, userName) {
        // Highlight selected researcher
        const items = document.querySelectorAll('#mobile-new-message-results .search-result-item');
        items.forEach(item => item.classList.remove('selected'));
        event.currentTarget.classList.add('selected');
        
        // Store selected user
        this.selectedUserForNewMessage = { id: userId, name: userName };
    },
    
    async sendNewMessage() {
        const message = document.getElementById('mobile-initial-message').value.trim();
        
        if (!this.selectedUserForNewMessage || !message) {
            this.showAlert('Please select a researcher and enter a message', 'warning');
            return;
        }
        
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'start_conversation',
                    user_id: this.selectedUserForNewMessage.id,
                    message: message
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                this.hideNewMessageModal();
                document.getElementById('mobile-initial-message').value = '';
                this.selectedUserForNewMessage = null;
                this.showAlert('Message sent successfully!', 'success');
                this.loadConversations(); // Refresh chat list
            } else {
                this.showAlert(data.message || 'Failed to send message', 'danger');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showAlert('Failed to send message', 'danger');
        }
    },
    
    searchResearchersForNewMessage(query) {
        const container = document.getElementById('mobile-new-message-results');
        const items = container.querySelectorAll('.search-result-item');
        
        if (!query.trim()) {
            // Show all when search is empty
            this.loadResearchersForNewMessage();
            return;
        }
        
        const searchTerm = query.toLowerCase();
        let hasVisibleItems = false;
        
        items.forEach(item => {
            const text = item.textContent.toLowerCase();
            if (text.includes(searchTerm)) {
                item.style.display = 'flex';
                hasVisibleItems = true;
            } else {
                item.style.display = 'none';
            }
        });
        
        if (!hasVisibleItems) {
            container.innerHTML = '<div class="text-center text-muted py-3">No researchers found</div>';
        }
    },
    
    async makeAdmin(userId) {
        if (!this.currentRoomId) return;
        
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'change_role',
                    room_id: this.currentRoomId,
                    user_id: userId,
                    role: 'admin'
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Update local data and refresh
                this.currentRoomParticipants.forEach(p => {
                    if (p.id === userId) {
                        p.role = 'admin';
                    }
                });
                this.showParticipantsModal(); // Refresh modal
                this.showAlert('User role updated successfully', 'success');
            } else {
                this.showAlert(data.message || 'Failed to update role', 'danger');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showAlert('Failed to update role', 'danger');
        }
    },
    
    async removeParticipant(userId, userName) {
        if (!this.currentRoomId || !confirm(`Are you sure you want to remove ${userName} from this chat?`)) {
            return;
        }
        
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'remove_participant',
                    room_id: this.currentRoomId,
                    user_id: userId
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Send system message
                await this.sendSystemMessage(`[SYSTEM] ${userName}|${userId} was removed from the conversation`);
                
                // Add system message to local messages immediately
                this.messages.push({
                    message: `[SYSTEM] ${userName}|${userId} was removed from the conversation`,
                    sender_id: 0,
                    created_at: new Date().toISOString(),
                    is_system: true
                });
                this.renderMessages();
                
                // Update local data
                this.currentRoomParticipants = this.currentRoomParticipants.filter(p => p.id !== userId);
                this.hideParticipantsModal();
                
                // Reload chat rooms to reflect changes
                await this.loadConversations();
                
                // If we removed ourselves, close the chat
                if (userId == <?php echo $userId; ?>) {
                    this.closeChat();
                } else {
                    // Update chat header
                    this.updateChatHeader();
                }
            } else {
                this.showAlert(data.message || 'Failed to remove participant', 'danger');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showAlert('Failed to remove participant', 'danger');
        }
    },
    
    async addParticipant(userId) {
        if (!this.currentRoomId) return;
        
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'add_participant',
                    room_id: this.currentRoomId,
                    user_id: userId
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Send system message about the new participant
                await this.sendSystemMessage(`[SYSTEM] ${data.participant_name}|${userId} was added to the conversation`);
                
                // Add system message to local messages immediately
                this.messages.push({
                    message: `[SYSTEM] ${data.participant_name}|${userId} was added to the conversation`,
                    sender_id: 0,
                    created_at: new Date().toISOString(),
                    is_system: true
                });
                this.renderMessages();
                
                this.hideAddParticipantsModal();
                
                // Refresh room details and chat list
                await this.loadRoomDetails();
                await this.loadConversations();
                this.updateChatHeader();
                
                // No alert message - silent addition
            } else {
                this.showAlert(data.message || 'Failed to add participant', 'danger');
            }
        } catch (error) {
            console.error('Error:', error);
            this.showAlert('Failed to add participant', 'danger');
        }
    },
    
    async sendSystemMessage(message) {
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'send_system_message',
                    room_id: this.currentRoomId,
                    message: message
                })
            });
            
            const data = await response.json();
            if (data.success) {
                // Success
            } else {
                console.error('Failed to send system message:', data.message);
            }
        } catch (error) {
            console.error('Error sending system message:', error);
        }
    },
    
    closeChat() {
        console.log('Closing chat');
        // Switch back to conversation list
        document.getElementById('chatView').classList.remove('active');
        document.getElementById('conversationListView').classList.remove('hidden');
        
        // Clear current room data
        this.currentRoomId = null;
        this.currentRoomName = '';
        this.currentRoomParticipants = [];
        this.messages = [];
    },
    
    async loadMessages() {
        try {
            const response = await fetch(`../api/chat.php?action=get_messages&room_id=${this.currentRoomId}&limit=50`);
            const data = await response.json();
            
            if (data.success) {
                this.messages = data.messages || [];
                this.renderMessages();
            } else {
                this.showEmptyMessages();
            }
        } catch (error) {
            this.showEmptyMessages();
        }
    },
    
    renderMessages() {
        const container = document.getElementById('messagesContainer');
        if (!container) {
            return;
        }
        
        if (this.messages.length === 0) {
            this.showEmptyMessages();
            return;
        }
        
        let html = '';
        for (let i = 0; i < this.messages.length; i++) {
            const message = this.messages[i];
            
            // Handle system messages
            if (message.is_system || message.sender_id == 0 || message.sender_id == null || message.message.startsWith('[SYSTEM]')) {
                const cleanMessage = message.message.replace('[SYSTEM] ', '');
                html += '<div class="system-message">';
                html += '<div class="system-message-text">' + this.processSystemMessage(cleanMessage) + '</div>';
                html += '</div>';
                continue;
            }
            
            const isOwn = message.sender_id == <?php echo $userId; ?>;
            
            html += '<div class="message-item ' + (isOwn ? 'sent' : 'received') + '">';
            html += '<div class="message-bubble">';
            if (!isOwn) {
                html += '<div class="message-sender">' + (message.sender_name || 'Unknown') + '</div>';
            }
            html += '<div class="message-text">' + message.message + '</div>';
            html += '<div class="message-time">' + this.formatTime(message.created_at) + '</div>';
            html += '</div>';
            html += '</div>';
        }
        
        container.innerHTML = html;
        
        this.scrollToBottom();
    },
    
    showEmptyMessages() {
        const container = document.getElementById('messagesContainer');
        if (container) {
            container.innerHTML = '<div class="empty-state"><i class="fas fa-comments"></i><h3>No messages yet</h3><p>Start the conversation!</p></div>';
        }
    },
    
    scrollToBottom() {
        const container = document.getElementById('messagesContainer');
        if (container) {
            container.scrollTop = container.scrollHeight;
        }
    },
    
    processSystemMessage(message) {
        // Handle system messages like "John Doe|123 was added to the conversation"
        if (message.includes('was added to the conversation')) {
            const nameWithId = message.replace(' was added to the conversation', '');
            const [name, id] = nameWithId.split('|');
            if (id) {
                return `<i class="fas fa-user-plus text-success me-1"></i><a href="<?php echo SITE_URL; ?>/public/researcher-profile.php?id=${id}" class="text-decoration-none fw-bold" style="color: inherit;">${name}</a> was added to the conversation`;
            } else {
                return `<i class="fas fa-user-plus text-success me-1"></i><strong>${name}</strong> was added to the conversation`;
            }
        }
        if (message.includes('was removed from the conversation')) {
            const nameWithId = message.replace(' was removed from the conversation', '');
            const [name, id] = nameWithId.split('|');
            if (id) {
                return `<i class="fas fa-user-minus text-danger me-1"></i><a href="<?php echo SITE_URL; ?>/public/researcher-profile.php?id=${id}" class="text-decoration-none fw-bold" style="color: inherit;">${name}</a> was removed from the conversation`;
            } else {
                return `<i class="fas fa-user-minus text-danger me-1"></i><strong>${name}</strong> was removed from the conversation`;
            }
        }
        return message;
    },
    
    formatTime(timestamp) {
        if (!timestamp) return '';
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    },
    
    sendMessage() {
        const input = document.getElementById('messageInput');
        const message = input.value.trim();
        
        if (!message || !this.currentRoomId) {
            return;
        }
        
        // Clear input immediately for better UX
        input.value = '';
        
        // Send the message
        this.sendMessageToAPI(message);
    },
    
    async sendMessageToAPI(message) {
        try {
            const response = await fetch('../api/chat.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    action: 'send_message',
                    room_id: this.currentRoomId,
                    message: message
                })
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            
            if (data.success) {
                // Reload messages to show the new one
                await this.loadMessages();
            } else {
                console.error('Failed to send message:', data.message);
                // Restore the message if sending failed
                document.getElementById('messageInput').value = message;
                this.showAlert('Failed to send message: ' + (data.message || 'Unknown error'), 'danger');
            }
        } catch (error) {
            console.error('Error sending message:', error);
            // Restore the message if sending failed
            document.getElementById('messageInput').value = message;
            this.showAlert('Failed to send message. Please try again.', 'danger');
        }
    },
    
    showAlert(message, type = 'info') {
        const container = document.getElementById('mobileAlertContainer');
        const alert = document.createElement('div');
        alert.className = `mobile-alert ${type}`;
        alert.innerHTML = `
            <span>${message}</span>
            <button type="button" class="btn-close" onclick="this.parentElement.remove()" style="background: none; border: none; font-size: 1.2rem; cursor: pointer;">&times;</button>
        `;
        container.appendChild(alert);
        
        // Auto-dismiss after 3 seconds
        setTimeout(() => {
            if (alert.parentElement) {
                alert.remove();
            }
        }, 3000);
    },
    
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    },
    
    getInitials(name) {
        if (!name) return '?';
        const parts = name.trim().split(' ');
        if (parts.length >= 2) {
            return (parts[0][0] + parts[1][0]).toUpperCase();
        } else if (parts.length === 1) {
            return parts[0][0].toUpperCase();
        }
        return '?';
    },
    
    stringToColor(str) {
        let hash = 0;
        for (let i = 0; i < str.length; i++) {
            hash = str.charCodeAt(i) + ((hash << 5) - hash);
        }
        let color = '#';
        for (let i = 0; i < 3; i++) {
            const value = (hash >> (i * 8)) & 0xFF;
            color += ('00' + value.toString(16)).substr(-2);
        }
        return color;
    },
    
    timeAgo(dateString) {
        const date = new Date(dateString);
        const now = new Date();
        const seconds = Math.floor((now - date) / 1000);
        
        const intervals = {
            year: 31536000,
            month: 2592000,
            week: 604800,
            day: 86400,
            hour: 3600,
            minute: 60
        };
        
        for (const [unit, secondsInUnit] of Object.entries(intervals)) {
            const interval = Math.floor(seconds / secondsInUnit);
            if (interval >= 1) {
                return interval === 1 ? `1 ${unit} ago` : `${interval} ${unit}s ago`;
            }
        }
        
        return 'Just now';
    },
    
    showNewChatModal() {
        this.showNewMessageModal();
    }
};

// Initialize on load
document.addEventListener('DOMContentLoaded', () => {
    console.log('DOM loaded, initializing MobileChatManager...');
    MobileChatManager.init();
});

// Global function for mobile compatibility
function showNewChatModal() {
    MobileChatManager.showNewChatModal();
}

// Add event delegation for better mobile touch handling
document.addEventListener('touchstart', function(e) {
    // Add active state to touched elements
    if (e.target.closest('.conversation-item')) {
        e.target.closest('.conversation-item').classList.add('active-touch');
    }
    if (e.target.closest('.chat-action-btn')) {
        e.target.closest('.chat-action-btn').classList.add('active-touch');
    }
    if (e.target.closest('.send-button')) {
        e.target.closest('.send-button').classList.add('active-touch');
    }
    if (e.target.closest('.search-result-item')) {
        e.target.closest('.search-result-item').classList.add('active-touch');
    }
    if (e.target.closest('.chat-user-info')) {
        e.target.closest('.chat-user-info').classList.add('active-touch');
    }
}, { passive: true });

document.addEventListener('touchend', function(e) {
    // Remove active state
    document.querySelectorAll('.active-touch').forEach(el => {
        el.classList.remove('active-touch');
    });
}, { passive: true });

// Prevent default behavior on touch moves for better scrolling
document.addEventListener('touchmove', function(e) {
    if (e.target.closest('.messages-container') || 
        e.target.closest('.mobile-conversation-list')) {
        // Allow natural scrolling
        return;
    }
}, { passive: true });

// Close modals when clicking outside
document.addEventListener('click', function(e) {
    const participantsModal = document.getElementById('participantsModalOverlay');
    const addParticipantsModal = document.getElementById('addParticipantsModalOverlay');
    
    if (participantsModal.classList.contains('show') && 
        !e.target.closest('.mobile-modal-content') && 
        !e.target.closest('.chat-user-info')) {
        MobileChatManager.hideParticipantsModal();
    }
    
    if (addParticipantsModal.classList.contains('show') && 
        !e.target.closest('.mobile-modal-content') && 
        !e.target.closest('.chat-action-btn')) {
        MobileChatManager.hideAddParticipantsModal();
    }
});

// Handle back button for Android
window.addEventListener('popstate', function(e) {
    const chatView = document.getElementById('chatView');
    if (chatView && chatView.classList.contains('active')) {
        MobileChatManager.closeChat();
        // Push state to prevent going back further
        history.pushState(null, null, window.location.href);
    }
});

// Initialize when page is fully loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        MobileChatManager.init();
    });
} else {
    MobileChatManager.init();
}
</script>

<?php include '../templates/researcher-mobile-footer.php'; ?>